<?php
/*
 * Elementor Varaus Service Widget
 * Author & Copyright: wpoceans
*/

namespace Elementor;

if (!defined('ABSPATH')) exit; // Exit if accessed directly

class Varaus_Service extends Widget_Base
{

	/**
	 * Retrieve the widget name.
	 */
	public function get_name()
	{
		return 'tmx-varaus_service';
	}

	/**
	 * Retrieve the widget title.
	 */
	public function get_title()
	{
		return esc_html__('Service', 'varaus-core');
	}

	/**
	 * Retrieve the widget icon.
	 */
	public function get_icon()
	{
		return 'eicon-archive';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 */
	public function get_categories()
	{
		return ['wpoceans-category'];
	}

	/**
	 * Retrieve the list of scripts the Varaus Service widget depended on.
	 * Used to set scripts dependencies required to run the widget.
	 */
	public function get_script_depends()
	{
		return ['tmx-varaus_service'];
	}

	/**
	 * Register Varaus Service widget controls.
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 */
	protected function register_controls()
	{


		$posts = get_posts('post_type="service"&numberposts=-1');
		$PostID = array();
		if ($posts) {
			foreach ($posts as $post) {
				$PostID[$post->ID] = $post->ID;
			}
		} else {
			$PostID[__('No ID\'s found', 'varaus')] = 0;
		}


		$this->start_controls_section(
			'section_service',
			[
				'label' => esc_html__('Service Options', 'varaus-core'),
			]
		);
		$this->add_control(
			'service_style',
			[
				'label' => esc_html__('Service Style', 'varaus-core'),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'style-one' => esc_html__('Style One', 'varaus-core'),
					'style-two' => esc_html__('Style Two', 'varaus-core'),
				],
				'default' => 'style-one',
				'description' => esc_html__('Select your service style.', 'varaus-core'),
			]
		);
		$this->end_controls_section(); // end: Section

		$this->start_controls_section(
			'section_service_listing',
			[
				'label' => esc_html__('Listing Options', 'varaus-core'),
			]
		);
		$this->add_control(
			'service_limit',
			[
				'label' => esc_html__('Service Limit', 'varaus-core'),
				'type' => Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 100,
				'step' => 1,
				'default' => 3,
				'description' => esc_html__('Enter the number of items to show.', 'varaus-core'),
			]
		);
		$this->add_control(
			'service_order',
			[
				'label' => __('Order', 'varaus-core'),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'ASC' => esc_html__('Asending', 'varaus-core'),
					'DESC' => esc_html__('Desending', 'varaus-core'),
				],
				'default' => 'DESC',
			]
		);
		$this->add_control(
			'service_orderby',
			[
				'label' => __('Order By', 'varaus-core'),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'none' => esc_html__('None', 'varaus-core'),
					'ID' => esc_html__('ID', 'varaus-core'),
					'author' => esc_html__('Author', 'varaus-core'),
					'title' => esc_html__('Title', 'varaus-core'),
					'date' => esc_html__('Date', 'varaus-core'),
				],
				'default' => 'date',
			]
		);
		$this->add_control(
			'service_show_category',
			[
				'label' => __('Certain Categories?', 'varaus-core'),
				'type' => Controls_Manager::SELECT2,
				'default' => [],
				'options' => Controls_Helper_Output::get_terms_names('service_category'),
				'multiple' => true,
			]
		);
		$this->add_control(
			'service_show_id',
			[
				'label' => __('Certain ID\'s?', 'varaus-core'),
				'type' => Controls_Manager::SELECT2,
				'default' => [],
				'options' => $PostID,
				'multiple' => true,
			]
		);
		$this->add_control(
			'short_content',
			[
				'label' => esc_html__('Excerpt Length', 'varaus-core'),
				'type' => Controls_Manager::NUMBER,
				'min' => 1,
				'step' => 1,
				'default' => 16,
				'description' => esc_html__('How many words you want in short content paragraph.', 'varaus-core'),
			]
		);

		$this->end_controls_section(); // end: Section


		// Icon
		$this->start_controls_section(
			'section_icons_style',
			[
				'label' => esc_html__('Icon', 'varaus-core'),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'varaus-core'),
				'name' => 'varaus_icons_typography',
				'selector' => '{{WRAPPER}} .service-area .service-item .fi:before',
			]
		);
		$this->add_control(
			'icons_color',
			[
				'label' => esc_html__('Color', 'varaus-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .service-area .service-item .fi:before' => 'color: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'icons_padding',
			[
				'label' => esc_html__('Title Padding', 'varaus-core'),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em'],
				'selectors' => [
					'{{WRAPPER}} .service-area .service-item .fi:before' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->end_controls_section(); // end: Section

		// Title
		$this->start_controls_section(
			'section_title_style',
			[
				'label' => esc_html__('service Title', 'varaus-core'),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'varaus-core'),
				'name' => 'varaus_title_typography',
				'selector' => '{{WRAPPER}} .service-area .service-item .fi:before',
			]
		);
		$this->add_control(
			'title_color',
			[
				'label' => esc_html__('Color', 'varaus-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .service-area .service-item .fi:before' => 'color: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'title_padding',
			[
				'label' => esc_html__('Title Padding', 'varaus-core'),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em'],
				'selectors' => [
					'{{WRAPPER}} .service-area .service-item .fi:before' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->end_controls_section(); // end: Section

		// Conent
		$this->start_controls_section(
			'section_content_style',
			[
				'label' => esc_html__('service Content', 'varaus-core'),
				'tab' => Controls_Manager::TAB_STYLE,
				'condition' => [
					'service_style' => array('style-one'),
				],
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'varaus-core'),
				'name' => 'varaus_content_typography',
				'selector' => '{{WRAPPER}} .service-area .service-text p',
			]
		);
		$this->add_control(
			'content_color',
			[
				'label' => esc_html__('Color', 'varaus-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .service-area .service-text p' => 'color: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'content_padding',
			[
				'label' => esc_html__('Title Padding', 'varaus-core'),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em'],
				'selectors' => [
					'{{WRAPPER}} .service-area .service-text p' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->end_controls_section(); // end: Section



	}

	/**
	 * Render Service widget output on the frontend.
	 * Written in PHP and used to generate the final HTML.
	 */
	protected function render()
	{
		$settings = $this->get_settings_for_display();

		$service_limit = !empty($settings['service_limit']) ? $settings['service_limit'] : '';

		$service_order = !empty($settings['service_order']) ? $settings['service_order'] : '';
		$service_orderby = !empty($settings['service_orderby']) ? $settings['service_orderby'] : '';
		$service_show_category = !empty($settings['service_show_category']) ? $settings['service_show_category'] : [];
		$service_show_id = !empty($settings['service_show_id']) ? $settings['service_show_id'] : [];
		$short_content = !empty($settings['short_content']) ? $settings['short_content'] : '';

		$excerpt_length = $short_content ? $short_content : '16';


		// Turn output buffer on
		ob_start();

		// Pagination
		global $paged;
		if (get_query_var('paged'))
			$my_page = get_query_var('paged');
		else {
			if (get_query_var('page'))
				$my_page = get_query_var('page');
			else
				$my_page = 1;
			set_query_var('paged', $my_page);
			$paged = $my_page;
		}

		if ($service_show_id) {
			$service_show_id = json_encode($service_show_id);
			$service_show_id = str_replace(array('[', ']'), '', $service_show_id);
			$service_show_id = str_replace(array('"', '"'), '', $service_show_id);
			$service_show_id = explode(',', $service_show_id);
		} else {
			$service_show_id = '';
		}

		$args = array(
			// other query params here,
			'paged' => $my_page,
			'post_type' => 'service',
			'posts_per_page' => (int)$service_limit,
			'service_category' => implode(',', $service_show_category),
			'orderby' => $service_orderby,
			'order' => $service_order,
			'post__in' => $service_show_id,
		);

		$varaus_service = new \WP_Query($args); ?>
		<div class="service-area row">
			<?php
			if ($varaus_service->have_posts()) : while ($varaus_service->have_posts()) : $varaus_service->the_post();

					$service_options = get_post_meta(get_the_ID(), 'service_options', true);
					$service_title = isset($service_options['service_title']) ? $service_options['service_title'] : '';
					$service_icon = isset($service_options['service_icon']) ? $service_options['service_icon'] : '';

			?>
					<div class="col col-md-4 col-sm-6 custom-grid col-12">
						<div class="service-item">
							<div class="service-icon">
								<?php if ($service_icon) {
									echo '<i class="fi ' . esc_attr($service_icon) . '"></i>';
								} ?>
							</div>
							<div class="service-text">
								<?php
								if ($service_title) {
									echo '<h2>' . esc_html($service_title) . '</h2>';
								}
								?>
								<p><?php echo wp_trim_words(get_the_content(), $excerpt_length, ' '); ?></p>
							</div>
						</div>
					</div>
			<?php
				endwhile;
			endif;
			wp_reset_postdata();
			?>
		</div>
<?php
		// Return outbut buffer
		echo ob_get_clean();
	}
	/**
	 * Render Service widget output in the editor.
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 */

	//protected function _content_template(){}

}
Plugin::instance()->widgets_manager->register(new Varaus_Service());
