<?php
/*
 * Elementor Varaus Room Widget
 * Author & Copyright: wpoceans
*/

namespace Elementor;

if (!defined('ABSPATH')) exit; // Exit if accessed directly

class Varaus_Room extends Widget_Base
{

	/**
	 * Retrieve the widget name.
	 */
	public function get_name()
	{
		return 'tmx-varaus_room';
	}

	/**
	 * Retrieve the widget title.
	 */
	public function get_title()
	{
		return esc_html__('Rooms', 'varaus-core');
	}

	/**
	 * Retrieve the widget icon.
	 */
	public function get_icon()
	{
		return 'eicon-posts-masonry';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 */
	public function get_categories()
	{
		return ['wpoceans-category'];
	}

	/**
	 * Retrieve the list of scripts the Varaus Room widget depended on.
	 * Used to set scripts dependencies required to run the widget.
	 */
	public function get_script_depends()
	{
		return ['wpo_varaus_rooms'];
	}

	/**
	 * Register Varaus Room widget controls.
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 */
	protected function register_controls()
	{


		$posts = get_posts('post_type="project"&numberposts=-1');
		$PostID = array();
		if ($posts) {
			foreach ($posts as $post) {
				$PostID[$post->ID] = $post->ID;
			}
		} else {
			$PostID[__('No ID\'s found', 'varaus')] = 0;
		}

		$this->start_controls_section(
			'section_room_listing',
			[
				'label' => esc_html__('Listing Options', 'varaus-core'),
			]
		);
		$this->add_control(
			'room_limit',
			[
				'label' => esc_html__('Room Limit', 'varaus-core'),
				'type' => Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 100,
				'step' => 1,
				'default' => 3,
				'description' => esc_html__('Enter the number of items to show.', 'varaus-core'),
			]
		);
		$this->add_control(
			'room_order',
			[
				'label' => __('Order', 'varaus-core'),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'ASC' => esc_html__('Asending', 'varaus-core'),
					'DESC' => esc_html__('Desending', 'varaus-core'),
				],
				'default' => 'DESC',
			]
		);
		$this->add_control(
			'room_orderby',
			[
				'label' => __('Order By', 'varaus-core'),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'none' => esc_html__('None', 'varaus-core'),
					'ID' => esc_html__('ID', 'varaus-core'),
					'author' => esc_html__('Author', 'varaus-core'),
					'title' => esc_html__('Title', 'varaus-core'),
					'date' => esc_html__('Date', 'varaus-core'),
				],
				'default' => 'date',
			]
		);
		$this->add_control(
			'room_show_category',
			[
				'label' => __('Certain Categories?', 'varaus-core'),
				'type' => Controls_Manager::SELECT2,
				'default' => [],
				'options' => Controls_Helper_Output::get_terms_names('hb_room_type'),
				'multiple' => true,
			]
		);
		$this->add_control(
			'room_show_id',
			[
				'label' => __('Certain ID\'s?', 'varaus-core'),
				'type' => Controls_Manager::SELECT2,
				'default' => [],
				'options' => $PostID,
				'multiple' => true,
			]
		);
		$this->end_controls_section(); // end: Section


		// Title
		$this->start_controls_section(
			'section_title_style',
			[
				'label' => esc_html__('Title', 'varaus-core'),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'varaus-core'),
				'name' => 'varaus_title_typography',
				'selector' => '{{WRAPPER}} .varaus-rooms-section h3 a',
			]
		);
		$this->add_control(
			'title_color',
			[
				'label' => esc_html__('Color', 'varaus-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .varaus-rooms-section h3 a ' => 'color: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'title_padding',
			[
				'label' => esc_html__('Title Padding', 'varaus-core'),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em'],
				'selectors' => [
					'{{WRAPPER}} .varaus-rooms-section h3 a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->end_controls_section(); // end: Section

		// Subtitle
		$this->start_controls_section(
			'section_subtitle_style',
			[
				'label' => esc_html__('Subtitle', 'varaus-core'),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'varaus-core'),
				'name' => 'varaus_subtitle_typography',
				'selector' => '{{WRAPPER}} .varaus-rooms-section .cat',
			]
		);
		$this->add_control(
			'subtitle_color',
			[
				'label' => esc_html__('Color', 'varaus-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .varaus-rooms-section .cat' => 'color: {{VALUE}};',
				],
			]
		);
		$this->end_controls_section(); // end: Section


	}

	/**
	 * Render Room widget output on the frontend.
	 * Written in PHP and used to generate the final HTML.
	 */
	protected function render()
	{
		$settings = $this->get_settings_for_display();
		$room_limit = !empty($settings['room_limit']) ? $settings['room_limit'] : '';
		$room_order = !empty($settings['room_order']) ? $settings['room_order'] : '';
		$room_orderby = !empty($settings['room_orderby']) ? $settings['room_orderby'] : '';
		$room_show_category = !empty($settings['room_show_category']) ? $settings['room_show_category'] : [];
		$room_show_id = !empty($settings['room_show_id']) ? $settings['room_show_id'] : [];

		// Turn output buffer on
		ob_start();

		// Pagination
		global $paged;
		if (get_query_var('paged'))
			$my_page = get_query_var('paged');
		else {
			if (get_query_var('page'))
				$my_page = get_query_var('page');
			else
				$my_page = 1;
			set_query_var('paged', $my_page);
			$paged = $my_page;
		}

		if ($room_show_id) {
			$room_show_id = json_encode($room_show_id);
			$room_show_id = str_replace(array('[', ']'), '', $room_show_id);
			$room_show_id = str_replace(array('"', '"'), '', $room_show_id);
			$room_show_id = explode(',', $room_show_id);
		} else {
			$room_show_id = '';
		}

		$args = array(
			// other query params here,
			'paged' => $my_page,
			'post_type' => 'hb_room',
			'posts_per_page' => (int)$room_limit,
			'hb_room_type' => implode(',', $room_show_category),
			'orderby' => $room_orderby,
			'order' => $room_order,
			'post__in' => $room_show_id,
		);

		$varaus_room = new \WP_Query($args);

		if ($varaus_room->have_posts()) : ?>
			<div class="varaus-rooms-section">
				<div class="rooms-grids clearfix">
					<?php
					while ($varaus_room->have_posts()) : $varaus_room->the_post();

						$room_options = get_post_meta(get_the_ID(), 'room_options', true);
						$room_image = isset($room_options['room_image']) ? $room_options['room_image'] : '';
						$starting_price = isset($room_options['starting_price']) ? $room_options['starting_price'] : '';

						global $post;
						$image_url = wp_get_attachment_url($room_image);
						$image_alt = get_post_meta($room_image, '_wp_attachment_image_alt', true);

					?>
						<div class="grid">
							<div class="img-holder">
								<?php if ($image_url) {
									echo '<img src="' . esc_url($image_url) . '" alt="' . esc_attr($image_alt) . '">';
								} ?>
							</div>
							<div class="details">
								<h3><a href="<?php echo esc_url(get_permalink()); ?>"><?php echo get_the_title(); ?></a></h3>
								<div class="cat">
									<span><?php echo esc_html($starting_price); ?></span>
									<?php echo esc_html__('/ Night ', 'varaus'); ?>
								</div>
							</div>
						</div>
					<?php endwhile;
					wp_reset_postdata(); ?>
				</div>
			</div>
<?php endif;
		// Return outbut buffer
		echo ob_get_clean();
	}
	/**
	 * Render Room widget output in the editor.
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 */

	//protected function _content_template(){}

}
Plugin::instance()->widgets_manager->register(new Varaus_Room());
